unit ManagedMethodExporterForm;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ComCtrls, ActnList, ImgList, StdCtrls;

type
  TfrmMain = class(TForm)
    edtAssembly: TEdit;
    btnLocateAssembly: TButton;
    memExports: TMemo;
    btnProcess: TButton;
    Label1: TLabel;
    ActionList: TActionList;
    ImageList: TImageList;
    actProcess: TAction;
    StatusBar1: TStatusBar;
    Label2: TLabel;
    dlgOpen: TOpenDialog;
    procedure actProcessExecute(Sender: TObject);
    procedure actProcessUpdate(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure btnLocateAssemblyClick(Sender: TObject);
  private
    Assembly,
    AssemblyFQ,
    AssemblyDir,
    AssemblyBackupFQ,
    TempDir,
    TempAssemblyFQ,
    TempIL,
    TempILFQ,
    TempRes: String;
    Methods: TStringList;
    Exported: Integer;
    procedure RunCommand(const Cmd, Args: String);
  public
    procedure CopyFile(const SourceFile, DestDir: String);
    procedure RenameFile(const SourceFile, DestFile: String);
    procedure DeleteFile(const TargetFile: String);
    procedure SetupVariables;
    procedure CopyAssemblyToTemp;
    procedure CopyTempAssemblyBack;
    procedure RunDisassembler;
    procedure ProcessExportedMethods;
    procedure RunAssembler;
  end;

var
  frmMain: TfrmMain;

implementation

{$R *.dfm}

uses
  ShellAPI, StrUtils;

procedure TfrmMain.actProcessExecute(Sender: TObject);
begin
  SetupVariables;
  CopyAssemblyToTemp;
  RunDisassembler;
  ProcessExportedMethods;
  RunAssembler;
  CopyTempAssemblyBack;
  MessageDlg(Format('Exported %d methods in:'#13#13'%s',
    [Exported, AssemblyFQ]), mtInformation, [mbOk], 0)
end;

procedure TfrmMain.actProcessUpdate(Sender: TObject);
begin
  (Sender as TAction).Enabled := FileExists(edtAssembly.Text)
end;

procedure TfrmMain.CopyAssemblyToTemp;
begin
  CopyFile(AssemblyFQ, TempDir);
end;

procedure TfrmMain.CopyFile(const SourceFile, DestDir: String);
var
  FOS: TSHFileOpStruct;
begin
  FOS.Wnd := Handle;
  FOS.wFunc := FO_COPY;
  FOS.pFrom := PChar(SourceFile + #0);
  FOS.pTo := PChar(DestDir);
  FOS.fFlags := FOF_NOCONFIRMATION or FOF_SILENT;
  SHFileOperation(FOS);
end;

procedure TfrmMain.RenameFile(const SourceFile, DestFile: String);
var
  FOS: TSHFileOpStruct;
begin
  FOS.Wnd := Handle;
  FOS.wFunc := FO_RENAME;
  FOS.pFrom := PChar(SourceFile + #0);
  FOS.pTo := PChar(DestFile);
  FOS.fFlags := FOF_NOCONFIRMATION or FOF_SILENT or FOF_RENAMEONCOLLISION;
  SHFileOperation(FOS);
end;

procedure TfrmMain.DeleteFile(const TargetFile: String);
var
  FOS: TSHFileOpStruct;
begin
  FOS.Wnd := Handle;
  FOS.wFunc := FO_DELETE;
  FOS.pFrom := PChar(TargetFile + #0);
  FOS.pTo := nil;
  FOS.fFlags := FOF_NOCONFIRMATION or FOF_SILENT;
  SHFileOperation(FOS);
end;

procedure TfrmMain.CopyTempAssemblyBack;
begin
  if FileExists(AssemblyBackupFQ) then
    DeleteFile(AssemblyBackupFQ);
  RenameFile(AssemblyFQ, AssemblyBackupFQ);
  CopyFile(TempAssemblyFQ, AssemblyDir);
end;

procedure TfrmMain.ProcessExportedMethods;
var
  IL: TStringList;
  Line, EndOfHeader, Method, DotPos: Integer;
  FullMethodName, RemainingMethodName, ThisMethodNamePart: String;

  function GetNextMethodNamePart: String;
  begin
    Result := '';
    if Length(RemainingMethodName) > 0 then
    begin
      DotPos := Pos('.', RemainingMethodName);
      if DotPos <> 0 then
      begin
        Result := Copy(RemainingMethodName, 1, DotPos - 1);
        Delete(RemainingMethodName, 1, DotPos);
      end
      else
      begin
        Result := RemainingMethodName;
        RemainingMethodName := ''
      end
    end
  end;

begin
  IL := TStringList.Create;
  try
    IL.LoadFromFile(TempILFQ);
    //Fix the header section first
    Line := 0;
    Exported := 0;
    while Line < IL.Count do
    begin
      if Pos('.corflags', IL[Line]) > 0 then
      begin
        if Pos('.corflags 0x00000002', IL[Line]) > 0 then
          raise Exception.Create(
            'Assembly looks like it has already been modified to export ' +
            'methods'#13'(the .corflags manifest directive has been modified)');
        IL[Line] := '.corflags 0x00000002'; //This will almost certainly be 0x00000001
        Inc(Line);
        IL.Insert(Line, Format('.data VT_01 = int32[%d]', [Methods.Count]));
        Inc(Line);
        IL.Insert(Line, Format('.vtfixup [%d] int32 fromunmanaged at VT_01', [Methods.Count]));
        Inc(Line);
        Break
      end;
      Inc(Line)
    end;
    EndOfHeader := Line;
    //Now deal with locating each method and amending that
    for Method := 0 to Methods.Count - 1 do
    begin
      Line := EndOfHeader;
      FullMethodName := Methods[Method];
      RemainingMethodName := FullMethodName;
      ThisMethodNamePart := GetNextMethodNamePart;
      if Length(ThisMethodNamePart) = 0 then
        raise Exception.CreateFmt('Invalid method name: %s', [FullMethodName]);
      repeat
        if (Pos(ThisMethodNamePart, IL[Line]) > 0) then
        begin
          if Length(RemainingMethodName) = 0 then
          begin
            Inc(Line, 2); //get past the {
            IL.Insert(Line, Format('      .vtentry 1:%d', [Method + 1]));
            Inc(Line);
            IL.Insert(Line, Format('      .export [%d] as %s', [Method + 1, ThisMethodNamePart]));
            Inc(Exported);
            Break
          end;
          ThisMethodNamePart := GetNextMethodNamePart;
        end;
        Inc(Line)
      until Line >= IL.Count;
    end;
    IL.SaveToFile(TempILFQ);
  finally
    IL.Free
  end
end;

procedure TfrmMain.RunCommand(const Cmd, Args: String);
var
  SEI: TShellExecuteInfo;
begin
  FillChar(SEI, SizeOf(SEI), 0);
  SEI.cbSize := SizeOf(SEI);
  SEI.fMask := SEE_MASK_NOCLOSEPROCESS;
  SEI.Wnd := Handle;
  SEI.lpFile := PChar(Cmd);
  SEI.lpParameters := PChar(Args);
  SEI.lpDirectory := PChar(TempDir);
  SEI.nShow := SW_SHOWNORMAL;
  ShellExecuteEx(@SEI);
  WaitForInputIdle(SEI.hProcess, Infinite);
  WaitForSingleObject(SEI.hProcess, Infinite);
  CloseHandle(SEI.hProcess);
end;

procedure TfrmMain.RunAssembler;
var
  CmdLineArgs: String;
begin
  CmdLineArgs := Format('/C ilasm %s /dll /quiet /nologo /debug /output:%s',
    [TempIL, Assembly]);
  RunCommand('cmd', CmdLineArgs);
end;

procedure TfrmMain.RunDisassembler;
var
  CmdLineArgs: String;
begin
  CmdLineArgs := Format('/C ildasm "%s" /linenum /nobar /out:"%s"',
    [AssemblyFQ, TempILFQ]);
  RunCommand('cmd', CmdLineArgs);
  DeleteFile(TempAssemblyFQ);
end;

procedure TfrmMain.SetupVariables;
var
  Buf: array[0..MAX_PATH] of Char;
  Name: PChar;
  I: Integer;
begin
  AssemblyFQ := ExpandFileName(edtAssembly.Text);
  Assembly := ExtractFileName(AssemblyFQ);
  AssemblyDir := IncludeTrailingPathDelimiter(ExtractFilePath(AssemblyFQ));
  AssemblyBackupFQ := ChangeFileExt(AssemblyFQ, '.old');
  GetTempPath(SizeOf(Buf), Buf);
  TempDir := IncludeTrailingPathDelimiter(Buf);
  TempAssemblyFQ := TempDir + Assembly;
  GetFullPathName(PChar(TempAssemblyFQ), SizeOf(Buf), Buf, Name);
  TempAssemblyFQ := Buf;
  TempRes := ChangeFileExt(Assembly, '.res');
  TempIL := ChangeFileExt(Assembly, '.il');
  TempILFQ := TempDir + TempIL;
  Methods.Clear;
  for I := memExports.Lines.Count - 1 downto 0 do
    if IsValidIdent(Trim(StringReplace(memExports.Lines[I], '.', 'A', [rfReplaceAll]))) then
      Methods.Insert(0, memExports.Lines[I])
    else
      if Trim(memExports.Lines[I]) <> '' then
        raise Exception.CreateFmt('%s is not a valid method name', [Trim(memExports.Lines[I])]);
end;

procedure TfrmMain.FormCreate(Sender: TObject);
begin
  Methods := TStringList.Create
end;

procedure TfrmMain.FormDestroy(Sender: TObject);
begin
  Methods.Free
end;

procedure TfrmMain.btnLocateAssemblyClick(Sender: TObject);
begin
  if dlgOpen.Execute then
    edtAssembly.Text := dlgOpen.FileName
end;

end.
